package gov.va.med.mhv.journal.service.validator;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.journal.data.model.ActivityJournal;
import gov.va.med.mhv.journal.dto.ActivityDetailDTO;
import gov.va.med.mhv.journal.dto.ActivityJournalDTO;
import gov.va.med.mhv.journal.repository.ActivityJournalRepository;
import gov.va.med.mhv.journal.web.util.JournalMessagesUtil;

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component 
public class ActivityJournalValidator {

	public static final int COMMENTS_MAX_LENGTH = 255;
	
	@Autowired
	private ActivityJournalRepository activityJournalRepository;
	
	@Autowired
	private JournalMessagesUtil messagesUtil;
	
	@Autowired
	private ActivityDetailValidator activityDetailValidator; 


	public ResponseUtil<ActivityJournalDTO> activityJournalValidations(ActivityJournalDTO activityJournalDto) {
		ResponseUtil<ActivityJournalDTO> response = new ResponseUtil<ActivityJournalDTO>();
		response = fieldValidation(activityJournalDto);
		
		// no errors in initial field validations, then perform complex
		// validations
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, activityJournalDto);
		}
		
		return response;
	}
	
	
	/**
	 * Validates Comments
	 */
	protected void validateComments(ActivityJournalDTO activityJournalDto, LinkedHashMap<String, String> validationErrors) {
		String value = StringUtils.trimToNull(activityJournalDto.getComments());
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
		//check max length
		Validations.validateMaxLength("Comments", value, COMMENTS_MAX_LENGTH, validationErrors);
	}	
	
	
	/**
	 * Validates Date
	 */
	protected void validateJournalDate(ActivityJournalDTO activityJournalDto, LinkedHashMap<String, String> validationErrors) {
		Date value = activityJournalDto.getJournalDate();
		//check requiredness:		
		if( value == null ) {
			Validations.validateRequired("JournalDate", null, validationErrors);
						
			//no need to continue:
			return;
		}		
	}	
	
	
	/**
	 * Validates modifiedDate
	 */
	protected void validateModifiedDate(ActivityJournalDTO activityJournalDto, LinkedHashMap<String, String> validationErrors) {
		Date value = activityJournalDto.getModifiedDate();
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
	}	
	
	/**
	 * Validates UserProfile
	 */
	protected void validateUserProfile(ActivityJournalDTO activityJournalDto, LinkedHashMap<String, String> validationErrors) {

		if (activityJournalDto.getUserId() == null) {
			//addError("userProfile", "null.not.allowed", new Object[] { "userProfile" });
			validationErrors.put("UserId", String.format(messagesUtil.getNullNotAllowed(), "UserProfileId"));
			// no need to continue:
			return;
		}

	}

	
	/**
	 * Executes all field-level validations
	 */
	protected ResponseUtil<ActivityJournalDTO> fieldValidation(ActivityJournalDTO activityJournalDto) {
		ResponseUtil<ActivityJournalDTO> response = new ResponseUtil<ActivityJournalDTO>();
		LinkedHashMap<String, String> validationErrors = new LinkedHashMap<String, String>();

		//validateGlassesOfWater(activityJournalDto, validationErrors);
		//validateCreatedDate(activityJournalDto, validationErrors);
		validateComments(activityJournalDto, validationErrors);
		validateJournalDate(activityJournalDto, validationErrors);
		//validateModifiedDate(activityJournalDto, validationErrors);

		//call field validation on children:
		if (null != activityJournalDto.getActivityDetails() && activityJournalDto.getActivityDetails().size() > 0) {
			for (ActivityDetailDTO activityDetailDTO : activityJournalDto.getActivityDetails()) {
				ResponseUtil<ActivityDetailDTO> activityDetailValidationResponse = activityDetailValidator.fieldValidation(activityDetailDTO);
				if (null != activityDetailValidationResponse.getValidationErrors() 
						&& activityDetailValidationResponse.getValidationErrors().size() > 0) {
					validationErrors.putAll(activityDetailValidationResponse.getValidationErrors());
				}
			}
		}
		
		if (activityJournalDto.getComments() == null && activityJournalDto.getComments().trim().length() == 0) {
					
			if (activityJournalDto.getActivityDesc().isEmpty() && activityJournalDto.getActivityDetails().size() == 0 ) {
				validationErrors.put("Activity Description is a required field", "Activity Description is a required field");
			}
		
			if (activityJournalDto.getActivityType().isEmpty() && activityJournalDto.getActivityDetails().size() == 0 ) {
				validationErrors.put("Activity Type is a required field", "Activity Type is a required field");
			}
		}	
		
		/**if (activityJournalDto.getActivityType()!= null && activityJournalDto.getActivityDetails().size() == 0 ) {
			validationErrors.put("Activity Type need to be Add to the Day", messagesUtil.getActivityTypeRequired());
		}
		**/
		
		validateInformationEntered(activityJournalDto, validationErrors);
		
		response.setValidationErrors(validationErrors);
		
		return response;
	}
	
	private void validateInformationEntered(ActivityJournalDTO activityJournalDto, LinkedHashMap<String, String> validationErrors) {
		if (StringUtils.trimToNull(activityJournalDto.getComments()) == null 
				&& (activityJournalDto.getActivityDetails() == null || (null != activityJournalDto.getActivityDetails() && activityJournalDto.getActivityDetails().size() ==0))) {
			validationErrors.put("sei.aj.noInformation", messagesUtil.getAjNoInformation());
		}
	}

	protected void complexValidation(ResponseUtil response, ActivityJournalDTO activityJournalDto) {
		LinkedHashMap<String, String> validationErrors = response.getValidationErrors();
		
		validateSingleEntryForDate(activityJournalDto, validationErrors);
		validateUserProfile(activityJournalDto, validationErrors);
		
		response.setValidationErrors(validationErrors);
	}
	
	/**
	 * Ensure that only one activity journal exists for a
	 * particular date
	 *
	 */
	private void validateSingleEntryForDate(ActivityJournalDTO activityJournalDto, LinkedHashMap<String, String> validationErrors) {
		if (activityJournalDto.getActivityJournalId() == null && null != activityJournalDto.getJournalDate()) {
			List<ActivityJournal> journals = activityJournalRepository.findActivityJournalsForUserAndDate(activityJournalDto.getUserId(), activityJournalDto.getJournalDate());
			
			if (journals.size() > 0) {
				validationErrors.put("singleEntryForDate", messagesUtil.getSeiFJDateAlreadyExists());
			}
		}
	}
}